import { useState, useEffect, useRef, useCallback } from 'react';
import { Ship } from '../utils/Ship';
import { Asteroid } from '../utils/Asteroid';
import { Bullet } from '../utils/Bullet';
import { Explosion } from '../utils/Explosion';

export const useGame = () => {
    const canvasRef = useRef(null);
    const animationRef = useRef(null);
    const [gameState, setGameState] = useState({
        lives: 5,
        round: 1,
        score: 0,
        gameActive: false,
        gameStarted: false,
        gamePaused: false,
        currentGameId: null,
        currentRound: 1,
        roundScore: 0,
        gameOverSaved: false,
        isRespawning: false,
        respawnTimer: 0,
        respawnDelay: 60,
        totalAsteroidsDestroyed: 0,
        roundAsteroidsDestroyed: 0,
        totalShotsFired: 0,
        roundShotsFired: 0,
        scaleFactor: 1
    });

    const [gameObjects, setGameObjects] = useState({
        ship: null,
        asteroids: [],
        bullets: [],
        explosions: []
    });

    const [images, setImages] = useState({
        coin: null,
        explosion: null
    });

    const [lastShotTime, setLastShotTime] = useState(0);
    const shotCooldown = 200;

    // Load images - using fallback colors instead
    useEffect(() => {
        // Create placeholder colored circles for asteroids
        const coinCanvas = document.createElement('canvas');
        coinCanvas.width = 64;
        coinCanvas.height = 64;
        const coinCtx = coinCanvas.getContext('2d');
        
        // Draw green hexagon for asteroid
        coinCtx.fillStyle = '#4CAF50';
        coinCtx.strokeStyle = '#66BB6A';
        coinCtx.lineWidth = 2;
        coinCtx.beginPath();
        for (let i = 0; i < 6; i++) {
            const angle = (Math.PI / 3) * i;
            const x = 32 + Math.cos(angle) * 20;
            const y = 32 + Math.sin(angle) * 20;
            if (i === 0) coinCtx.moveTo(x, y);
            else coinCtx.lineTo(x, y);
        }
        coinCtx.closePath();
        coinCtx.fill();
        coinCtx.stroke();
        
        const explosionCanvas = document.createElement('canvas');
        explosionCanvas.width = 64;
        explosionCanvas.height = 64;
        const explosionCtx = explosionCanvas.getContext('2d');
        
        // Draw explosion particles
        for (let i = 0; i < 20; i++) {
            const x = 32 + (Math.random() - 0.5) * 40;
            const y = 32 + (Math.random() - 0.5) * 40;
            const size = Math.random() * 8 + 2;
            
            explosionCtx.fillStyle = `hsl(${Math.random() * 60}, 100%, ${50 + Math.random() * 50}%)`;
            explosionCtx.beginPath();
            explosionCtx.arc(x, y, size, 0, Math.PI * 2);
            explosionCtx.fill();
        }
        
        // Convert canvases to images
        const coinImage = new Image();
        const explosionImage = new Image();
        
        coinImage.onload = () => {
            setImages(prev => ({ ...prev, coin: coinImage }));
        };
        
        explosionImage.onload = () => {
            setImages(prev => ({ ...prev, explosion: explosionImage }));
        };
        
        coinImage.src = coinCanvas.toDataURL();
        explosionImage.src = explosionCanvas.toDataURL();
    }, []);

    // Initialize ship
    useEffect(() => {
        if (canvasRef.current && !gameObjects.ship) {
            const canvas = canvasRef.current;
            const ship = new Ship(canvas.width / 2, canvas.height / 2, gameState.scaleFactor);
            setGameObjects(prev => ({ ...prev, ship }));
        }
    }, [canvasRef.current, gameObjects.ship, gameState.scaleFactor]);

    // Setup responsive canvas
    const setupResponsiveCanvas = useCallback(() => {
        if (!canvasRef.current) return;

        const canvas = canvasRef.current;
        const baseWidth = 1920;
        const baseHeight = 1080;
        const baseCanvasSize = 1000;
        
        const windowWidth = window.innerWidth;
        const windowHeight = window.innerHeight;
        
        const scaleFactor = Math.min(windowWidth / baseWidth, windowHeight / baseHeight);
        let canvasSize = Math.floor(baseCanvasSize * scaleFactor);
        
        const minSize = Math.min(300, windowWidth * 0.6);
        const maxSize = Math.min(windowWidth * 0.9, windowHeight * 0.9);
        
        canvasSize = Math.max(minSize, Math.min(canvasSize, maxSize));
        
        const objectScaleFactor = canvasSize / baseCanvasSize;
        
        canvas.width = canvasSize;
        canvas.height = canvasSize;
        
        setGameState(prev => ({ ...prev, scaleFactor: objectScaleFactor }));
        
        if (gameObjects.ship) {
            gameObjects.ship.radius = 15 * objectScaleFactor;
            gameObjects.ship.size = 30 * objectScaleFactor;
        }
    }, [gameObjects.ship]);

    useEffect(() => {
        setupResponsiveCanvas();
        window.addEventListener('resize', setupResponsiveCanvas);
        window.addEventListener('orientationchange', () => {
            setTimeout(setupResponsiveCanvas, 100);
        });
        
        return () => {
            window.removeEventListener('resize', setupResponsiveCanvas);
            window.removeEventListener('orientationchange', setupResponsiveCanvas);
        };
    }, [setupResponsiveCanvas]);

    // Spawn asteroids
    const spawnAsteroids = useCallback((count) => {
        if (!images.coin) return;
        
        const newAsteroids = [];
        const canvas = canvasRef.current;
        const ship = gameObjects.ship;
        
        for (let i = 0; i < count; i++) {
            let x, y;
            let attempts = 0;
            const maxAttempts = 50;
            const minDistance = 150 * gameState.scaleFactor;
            
            // Try to spawn asteroid away from ship
            do {
                x = Math.random() * canvas.width;
                y = Math.random() * canvas.height;
                attempts++;
                
                if (attempts >= maxAttempts) break; // Prevent infinite loop
                
            } while (ship && 
                    Math.sqrt((x - ship.x) ** 2 + (y - ship.y) ** 2) < minDistance);
            
            newAsteroids.push(new Asteroid(
                x, y,
                'large',
                images.coin,
                gameState.scaleFactor
            ));
        }
        setGameObjects(prev => ({ ...prev, asteroids: [...prev.asteroids, ...newAsteroids] }));
    }, [images.coin, gameState.scaleFactor, gameObjects.ship]);

    // Fire bullet
    const fireBullet = useCallback(() => {
        if (!gameState.gameActive || gameState.gamePaused || gameState.isRespawning) return;
        
        const currentTime = Date.now();
        if (currentTime - lastShotTime < shotCooldown) return;
        
        if (gameObjects.ship) {
            const newBullet = new Bullet(
                gameObjects.ship.x,
                gameObjects.ship.y,
                gameObjects.ship.angle,
                gameState.scaleFactor
            );
            
            setGameObjects(prev => ({ ...prev, bullets: [...prev.bullets, newBullet] }));
            setLastShotTime(currentTime);
            
            setGameState(prev => ({
                ...prev,
                totalShotsFired: prev.totalShotsFired + 1,
                roundShotsFired: prev.roundShotsFired + 1
            }));
        }
    }, [gameState, lastShotTime, gameObjects.ship, gameState.scaleFactor]);

    // Check collisions
    const checkCollisions = useCallback(() => {
        setGameObjects(prev => {
            const newState = { ...prev };
            let scoreIncrease = 0;
            let asteroidsDestroyed = 0;

            // Bullet-asteroid collisions
            for (let i = newState.bullets.length - 1; i >= 0; i--) {
                for (let j = newState.asteroids.length - 1; j >= 0; j--) {
                    const bullet = newState.bullets[i];
                    const asteroid = newState.asteroids[j];
                    
                    if (checkCollision(bullet, asteroid)) {
                        switch(asteroid.size) {
                            case 'large': scoreIncrease += 20; break;
                            case 'medium': scoreIncrease += 30; break;
                            case 'small': scoreIncrease += 50; break;
                        }
                        
                        asteroidsDestroyed++;
                        
                        // Create explosion
                        const explosion = new Explosion(asteroid.x, asteroid.y);
                        newState.explosions.push(explosion);
                        
                        // Split asteroid if not small
                        if (asteroid.size !== 'small') {
                            const newSize = asteroid.size === 'large' ? 'medium' : 'small';
                            if (images.coin) {
                                newState.asteroids.push(new Asteroid(asteroid.x, asteroid.y, newSize, images.coin, gameState.scaleFactor));
                                newState.asteroids.push(new Asteroid(asteroid.x, asteroid.y, newSize, images.coin, gameState.scaleFactor));
                            }
                        }
                        
                        newState.bullets.splice(i, 1);
                        newState.asteroids.splice(j, 1);
                        break;
                    }
                }
            }

            // Ship-asteroid collisions
            if (newState.ship && newState.ship.visible) {
                for (let i = newState.asteroids.length - 1; i >= 0; i--) {
                    const asteroid = newState.asteroids[i];
                    if (checkCollision(newState.ship, asteroid)) {
                        // Make ship invisible immediately
                        newState.ship.visible = false;
                        
                        // Create explosion
                        createShipExplosion(newState.ship.x, newState.ship.y, newState.explosions);
                        newState.asteroids.splice(i, 1);
                        
                        // Trigger life loss immediately
                        loseLife();
                        return newState;
                    }
                }
            }

            if (scoreIncrease > 0) {
                setGameState(prev => ({
                    ...prev,
                    score: prev.score + scoreIncrease,
                    roundScore: prev.roundScore + scoreIncrease,
                    totalAsteroidsDestroyed: prev.totalAsteroidsDestroyed + asteroidsDestroyed,
                    roundAsteroidsDestroyed: prev.roundAsteroidsDestroyed + asteroidsDestroyed
                }));
            }

            return newState;
        });
    }, [images.coin, gameState.scaleFactor]);

    const checkCollision = (obj1, obj2) => {
        const dx = obj1.x - obj2.x;
        const dy = obj1.y - obj2.y;
        const distance = Math.sqrt(dx * dx + dy * dy);
        return distance < obj1.radius + obj2.radius;
    };

    const createShipExplosion = (x, y, explosions) => {
        const shipExplosion = {
            x, y,
            particles: [],
            shockwaves: [],
            debris: [],
            finished: false,
            age: 0,
            maxAge: 90,
            
            update() {
                this.age++;
                
                for (let i = this.particles.length - 1; i >= 0; i--) {
                    const particle = this.particles[i];
                    particle.x += particle.vx;
                    particle.y += particle.vy;
                    particle.vx *= 0.98;
                    particle.vy *= 0.98;
                    particle.life -= particle.decay;
                    particle.size *= 0.97;
                    
                    if (particle.life <= 0 || particle.size < 0.1) {
                        this.particles.splice(i, 1);
                    }
                }
                
                for (let i = this.debris.length - 1; i >= 0; i--) {
                    const piece = this.debris[i];
                    piece.x += piece.vx;
                    piece.y += piece.vy;
                    piece.vx *= 0.99;
                    piece.vy *= 0.99;
                    piece.angle += piece.rotationSpeed;
                    piece.life -= 0.01;
                    
                    if (piece.life <= 0) {
                        this.debris.splice(i, 1);
                    }
                }
                
                for (let i = this.shockwaves.length - 1; i >= 0; i--) {
                    const wave = this.shockwaves[i];
                    wave.radius += wave.speed;
                    wave.opacity -= 0.015;
                    
                    if (wave.opacity <= 0) {
                        this.shockwaves.splice(i, 1);
                    }
                }
                
                if (this.age > this.maxAge && this.particles.length === 0 && this.debris.length === 0) {
                    this.finished = true;
                }
            },
            
            draw(ctx) {
                ctx.save();
                
                this.shockwaves.forEach(wave => {
                    ctx.globalAlpha = wave.opacity;
                    ctx.strokeStyle = wave.color;
                    ctx.lineWidth = 3;
                    ctx.shadowColor = wave.color;
                    ctx.shadowBlur = 15;
                    
                    ctx.beginPath();
                    ctx.arc(this.x, this.y, wave.radius, 0, Math.PI * 2);
                    ctx.stroke();
                });
                
                this.debris.forEach(piece => {
                    ctx.save();
                    ctx.globalAlpha = piece.life;
                    ctx.translate(piece.x, piece.y);
                    ctx.rotate(piece.angle);
                    ctx.fillStyle = piece.color;
                    ctx.strokeStyle = '#ffffff';
                    ctx.lineWidth = 1;
                    
                    ctx.beginPath();
                    ctx.moveTo(piece.points[0].x, piece.points[0].y);
                    piece.points.forEach(point => {
                        ctx.lineTo(point.x, point.y);
                    });
                    ctx.closePath();
                    ctx.fill();
                    ctx.stroke();
                    
                    ctx.restore();
                });
                
                this.particles.forEach(particle => {
                    ctx.globalAlpha = particle.life;
                    ctx.fillStyle = particle.color;
                    ctx.shadowColor = particle.color;
                    ctx.shadowBlur = 15;
                    
                    ctx.beginPath();
                    ctx.arc(particle.x, particle.y, particle.size, 0, Math.PI * 2);
                    ctx.fill();
                });
                
                ctx.restore();
            }
        };
        
        // Create explosion particles
        const particleCount = 40 + Math.random() * 20;
        for (let i = 0; i < particleCount; i++) {
            const angle = (Math.PI * 2 * i) / particleCount + Math.random() * 0.5;
            const speed = 3 + Math.random() * 6;
            const size = 2 + Math.random() * 4;
            
            const particleType = Math.random();
            let color;
            
            if (particleType < 0.2) {
                color = '#ffffff';
            } else if (particleType < 0.4) {
                color = '#00ffff';
            } else if (particleType < 0.7) {
                color = `hsl(${20 + Math.random() * 40}, 100%, ${50 + Math.random() * 30}%)`;
            } else {
                color = `hsl(${0 + Math.random() * 20}, 100%, ${40 + Math.random() * 20}%)`;
            }
            
            shipExplosion.particles.push({
                x, y,
                vx: Math.cos(angle) * speed,
                vy: Math.sin(angle) * speed,
                size, color,
                life: 1,
                decay: 0.01 + Math.random() * 0.02
            });
        }
        
        // Create ship debris
        const debrisCount = 8 + Math.random() * 4;
        for (let i = 0; i < debrisCount; i++) {
            const angle = Math.random() * Math.PI * 2;
            const speed = 1 + Math.random() * 3;
            
            const points = [];
            const pointCount = 3 + Math.floor(Math.random() * 3);
            for (let j = 0; j < pointCount; j++) {
                points.push({
                    x: (Math.random() - 0.5) * 15,
                    y: (Math.random() - 0.5) * 15
                });
            }
            
            shipExplosion.debris.push({
                x, y,
                vx: Math.cos(angle) * speed,
                vy: Math.sin(angle) * speed,
                angle: Math.random() * Math.PI * 2,
                rotationSpeed: (Math.random() - 0.5) * 0.2,
                points, color: `hsl(${200 + Math.random() * 60}, 70%, ${40 + Math.random() * 30}%)`,
                life: 1
            });
        }
        
        // Create shockwaves
        for (let i = 0; i < 3; i++) {
            shipExplosion.shockwaves.push({
                radius: 10 + i * 15,
                speed: 3 + i * 0.5,
                opacity: 1 - i * 0.2,
                color: i === 0 ? '#ffffff' : i === 1 ? '#00ffff' : '#ffaa00'
            });
        }
        
        explosions.push(shipExplosion);
    };

    // Game loop
    const gameLoop = useCallback(() => {
        if (!gameState.gameActive || gameState.gamePaused) return;
        
        const canvas = canvasRef.current;
        const ctx = canvas?.getContext('2d');
        if (!ctx || !canvas) return;

        // Clear canvas
        ctx.fillStyle = 'black';
        ctx.fillRect(0, 0, canvas.width, canvas.height);

        // Update respawn timer
        if (gameState.isRespawning) {
            setGameState(prev => {
                const newRespawnTimer = prev.respawnTimer - 1;
                if (newRespawnTimer <= 0) {
                    if (gameObjects.ship) {
                        gameObjects.ship.reset(canvas.width, canvas.height);
                        gameObjects.ship.visible = true;
                    }
                    return { ...prev, respawnTimer: 0, isRespawning: false };
                }
                return { ...prev, respawnTimer: newRespawnTimer };
            });
        }

        // Update ship
        if (gameObjects.ship && !gameState.isRespawning) {
            gameObjects.ship.update(canvas.width, canvas.height);
        }

        // Update bullets
        setGameObjects(prev => ({
            ...prev,
            bullets: prev.bullets.filter(bullet => {
                bullet.update();
                return bullet.x > -50 && bullet.x < canvas.width + 50 && 
                       bullet.y > -50 && bullet.y < canvas.height + 50 &&
                       !bullet.isExpired();
            })
        }));

        // Update asteroids
        gameObjects.asteroids.forEach(asteroid => asteroid.update(canvas.width, canvas.height));

        // Update explosions
        setGameObjects(prev => ({
            ...prev,
            explosions: prev.explosions.filter(explosion => {
                explosion.update();
                return !explosion.finished;
            })
        }));

        // Remove expired small asteroids
        setGameObjects(prev => ({
            ...prev,
            asteroids: prev.asteroids.filter(asteroid => {
                if (asteroid.size === 'small') {
                    const currentTime = Date.now();
                    return currentTime - asteroid.creationTime <= 15000;
                }
                return true;
            })
        }));

        // Check collisions
        checkCollisions();

        // Check round completion
        if (gameObjects.asteroids.length === 0) {
            nextRound();
        }

        // Draw everything
        if (gameObjects.ship && gameObjects.ship.visible) {
            gameObjects.ship.draw(ctx);
        }
        gameObjects.asteroids.forEach(asteroid => asteroid.draw(ctx));
        gameObjects.bullets.forEach(bullet => bullet.draw(ctx));
        gameObjects.explosions.forEach(explosion => explosion.draw(ctx));

        animationRef.current = requestAnimationFrame(gameLoop);
    }, [gameState, gameObjects, checkCollisions]);

    // Start game
    const startGame = useCallback(() => {
        setGameState(prev => ({
            ...prev,
            gameActive: true,
            gameStarted: true
        }));
        spawnAsteroids(4);
    }, [spawnAsteroids]);

    // Next round
    const nextRound = useCallback(() => {
        setGameState(prev => ({
            ...prev,
            currentRound: prev.currentRound + 1,
            round: prev.round + 1,
            roundScore: 0,
            roundAsteroidsDestroyed: 0,
            roundShotsFired: 0
        }));
        spawnAsteroids(4 + gameState.round);
    }, [spawnAsteroids, gameState.round]);

    // Lose life
    const loseLife = useCallback(() => {
        setGameState(prev => {
            const newLives = prev.lives - 1;
            if (newLives <= 0) {
                gameOver();
                return prev;
            }
            
            return {
                ...prev,
                lives: newLives,
                isRespawning: true,
                respawnTimer: prev.respawnDelay
            };
        });
        
        if (gameObjects.ship) {
            gameObjects.ship.visible = false;
        }
    }, [gameObjects.ship]);

    // Game over
    const gameOver = useCallback(() => {
        setGameState(prev => ({ ...prev, gameActive: false }));
    }, []);

    // Toggle pause
    const togglePause = useCallback(() => {
        setGameState(prev => ({ ...prev, gamePaused: !prev.gamePaused }));
    }, []);

    // Handle keyboard input
    useEffect(() => {
        const handleKeyDown = (e) => {
            if (!gameState.gameActive) return;
            
            if (e.key === 'p' || e.key === 'P') {
                togglePause();
                return;
            }
            
            if (gameState.gamePaused) return;
            
            if (gameObjects.ship) {
                switch(e.key) {
                    case 'ArrowLeft': 
                        gameObjects.ship.rotateLeft = true; 
                        e.preventDefault();
                        break;
                    case 'ArrowRight': 
                        gameObjects.ship.rotateRight = true; 
                        e.preventDefault();
                        break;
                    case 'ArrowUp': 
                        gameObjects.ship.thrusting = true; 
                        e.preventDefault();
                        break;
                    case ' ': 
                        fireBullet(); 
                        e.preventDefault();
                        break;
                }
            }
        };

        const handleKeyUp = (e) => {
            if (!gameState.gameActive) return;
            
            if (gameObjects.ship) {
                switch(e.key) {
                    case 'ArrowLeft': 
                        gameObjects.ship.rotateLeft = false; 
                        e.preventDefault();
                        break;
                    case 'ArrowRight': 
                        gameObjects.ship.rotateRight = false; 
                        e.preventDefault();
                        break;
                    case 'ArrowUp': 
                        gameObjects.ship.thrusting = false; 
                        e.preventDefault();
                        break;
                }
            }
        };

        window.addEventListener('keydown', handleKeyDown);
        window.addEventListener('keyup', handleKeyUp);

        return () => {
            window.removeEventListener('keydown', handleKeyDown);
            window.removeEventListener('keyup', handleKeyUp);
        };
    }, [gameState, gameObjects.ship, fireBullet, togglePause]);

    // Start game loop when game becomes active
    useEffect(() => {
        if (gameState.gameActive && !gameState.gamePaused) {
            animationRef.current = requestAnimationFrame(gameLoop);
        }
        
        return () => {
            if (animationRef.current) {
                cancelAnimationFrame(animationRef.current);
            }
        };
    }, [gameState.gameActive, gameState.gamePaused, gameLoop]);

    return {
        canvasRef,
        gameState,
        gameObjects,
        startGame,
        fireBullet,
        togglePause,
        loseLife,
        gameOver
    };
};
