<?php
/**
 * Database Schema Verification Script
 */

require_once 'Database.php';

try {
    $db = Database::getInstance();
    $connection = $db->getConnection();
    
    echo "Checking database tables...\n\n";
    
    // List of required tables
    $requiredTables = [
        'contact_submissions',
        'ip_tracking', 
        'email_tracking',
        'security_events'
    ];
    
    // Check each table
    foreach ($requiredTables as $table) {
        $result = $db->fetch("SHOW TABLES LIKE ?", [$table]);
        if ($result) {
            echo "✓ Table '$table' exists\n";
        } else {
            echo "✗ Table '$table' is MISSING\n";
        }
    }
    
    // Check for views
    echo "\nChecking views...\n";
    $requiredViews = [
        'recent_submissions',
        'submission_stats'
    ];
    
    foreach ($requiredViews as $view) {
        $result = $db->fetch("SHOW TABLES LIKE ?", [$view]);
        if ($result) {
            echo "✓ View '$view' exists\n";
        } else {
            echo "✗ View '$view' is MISSING\n";
        }
    }
    
    // If missing, create them
    echo "\nCreating missing objects...\n";
    
    // Create recent_submissions view
    try {
        $db->exec("
            CREATE OR REPLACE VIEW recent_submissions AS 
            SELECT * FROM contact_submissions 
            ORDER BY created_at DESC LIMIT 50
        ");
        echo "✓ Created recent_submissions view\n";
    } catch (Exception $e) {
        echo "✗ Failed to create recent_submissions view: " . $e->getMessage() . "\n";
    }
    
    // Create submission_stats view
    try {
        $db->exec("
            CREATE OR REPLACE VIEW submission_stats AS 
            SELECT 
                DATE(created_at) as submission_date,
                COUNT(*) as total_submissions,
                SUM(CASE WHEN submission_status = 'processed' THEN 1 ELSE 0 END) as processed_submissions,
                SUM(CASE WHEN submission_status = 'pending' THEN 1 ELSE 0 END) as pending_submissions,
                SUM(CASE WHEN submission_status = 'error' THEN 1 ELSE 0 END) as error_submissions
            FROM contact_submissions 
            GROUP BY DATE(created_at)
            ORDER BY submission_date DESC
        ");
        echo "✓ Created submission_stats view\n";
    } catch (Exception $e) {
        echo "✗ Failed to create submission_stats view: " . $e->getMessage() . "\n";
    }
    
    // Create stored procedure for IP tracking
    try {
        $db->exec("
            DROP PROCEDURE IF EXISTS update_ip_tracking
        ");
        
        $db->exec("
            CREATE PROCEDURE update_ip_tracking(
                IN ip_address VARCHAR(45),
                IN country_code VARCHAR(2),
                IN city VARCHAR(100),
                IN organization VARCHAR(255)
            )
            BEGIN
                INSERT INTO ip_tracking (
                    ip_address, country_code, city, organization, 
                    total_submissions, reputation_score, created_at, updated_at
                ) VALUES (
                    ip_address, country_code, city, organization,
                    1, 1.0, NOW(), NOW()
                )
                ON DUPLICATE KEY UPDATE
                    total_submissions = total_submissions + 1,
                    reputation_score = CASE 
                        WHEN total_submissions > 20 THEN 0.2
                        WHEN total_submissions > 10 THEN 0.5
                        ELSE 1.0
                    END,
                    country_code = IFNULL(country_code, ip_tracking.country_code),
                    city = IFNULL(city, ip_tracking.city),
                    organization = IFNULL(organization, ip_tracking.organization),
                    updated_at = NOW();
            END
        ");
        echo "✓ Created update_ip_tracking stored procedure\n";
    } catch (Exception $e) {
        echo "✗ Failed to create stored procedure: " . $e->getMessage() . "\n";
    }
    
    echo "\nDatabase schema verification complete!\n";
    
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
}
?>
