/**
 * Phone Number Protection System
 * SEO-optimized phone number obfuscation to prevent spam bots
 */

class PhoneProtection {
    private $phoneNumber = '6173026011';
    private $formattedPhone = '+1 (617) 302-6011';
    
    /**
     * Generate obfuscated phone number for display
     */
    public function getObfuscatedPhone() {
        // Split into parts for JavaScript reconstruction
        $parts = [
            substr($this->phoneNumber, 0, 3),    // 617
            substr($this->phoneNumber, 3, 3),    // 302
            substr($this->phoneNumber, 6, 4)     // 6011
        ];
        
        return [
            'parts' => $parts,
            'display' => $this->formattedPhone,
            'tel_link' => 'tel:' . $this->phoneNumber
        ];
    }
    
    /**
     * Generate CSS-based obfuscated HTML
     */
    public function getProtectedPhoneHTML() {
        $data = $this->getObfuscatedPhone();
        
        return '<span class="phone-protected" data-phone-parts="' . htmlspecialchars(json_encode($data['parts'])) . '" data-tel="' . htmlspecialchars($data['tel_link']) . '">' .
               '<span class="phone-placeholder">Click to reveal phone number</span>' .
               '<noscript><span style="unicode-bidi:bidi-override; direction:rtl;">' . strrev($this->formattedPhone) . '</span></noscript>' .
               '</span>';
    }
    
    /**
     * Generate JavaScript for phone number reconstruction
     */
    public function getProtectionScript() {
        return '
        <script>
        document.addEventListener("DOMContentLoaded", function() {
            const phoneElements = document.querySelectorAll(".phone-protected");
            
            phoneElements.forEach(function(element) {
                const parts = JSON.parse(element.dataset.phoneParts);
                const telLink = element.dataset.tel;
                
                // Reconstruct phone number
                const phoneNumber = parts.join("");
                const formattedNumber = "+1 (" + parts[0] + ") " + parts[1] + "-" + parts[2];
                
                // Create clickable phone link
                const phoneLink = document.createElement("a");
                phoneLink.href = telLink;
                phoneLink.className = "text-white phone-revealed";
                phoneLink.innerHTML = "<i class=\"fas fa-phone me-2\"></i>" + formattedNumber;
                phoneLink.style.textDecoration = "none";
                
                // Replace placeholder with actual phone number
                const placeholder = element.querySelector(".phone-placeholder");
                if (placeholder) {
                    placeholder.replaceWith(phoneLink);
                }
                
                // Add click tracking for analytics (optional)
                phoneLink.addEventListener("click", function() {
                    if (typeof gtag !== "undefined") {
                        gtag("event", "phone_click", {
                            "event_category": "contact",
                            "event_label": "footer_phone"
                        });
                    }
                });
            });
        });
        </script>';
    }
}

// Create global instance
$phoneProtection = new PhoneProtection();
